package net.hurstfrost.santa.web;

import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.hurstfrost.santa.control.SantaControlService;
import net.hurstfrost.santa.control.SantaControlServiceImpl;
import net.hurstfrost.santa.sound.SantaSoundService;
import net.hurstfrost.santa.sound.SantaSoundServiceImpl;
import net.hurstfrost.santa.sound.SantaVoice;
import net.hurstfrost.santa.sound.SoundBiteException;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import twitter4j.org.json.JSONObject;

import com.caucho.burlap.server.BurlapServlet;

public class SantaServlet extends BurlapServlet implements SantaControlService, SantaSoundService {
	public static class LogEntry {

		private final String remoteAddress;
		private final Activity activity;
		private final String[] extraInfo;
		private Date date;
		private final String queryString;
		private DateFormat dateFormat;

		public LogEntry(String remoteAddress, Activity activity, String queryString, String[] extraInfo) {
			this.queryString = queryString;
			this.extraInfo = extraInfo;
			this.activity = activity;
			this.remoteAddress = remoteAddress;
			date = new Date();
			dateFormat = SimpleDateFormat.getDateTimeInstance(SimpleDateFormat.SHORT, SimpleDateFormat.SHORT);
		}
		
		@Override
		public String toString() {
			String	log = String.format("%s : %s : %s%s", dateFormat.format(date), remoteAddress, activity, queryString==null?"":"?"+queryString);
			
			if (!ArrayUtils.isEmpty(extraInfo)) {
				log += " ( ";
				for (String extra : extraInfo) {
					log += "+" + extra + " ";
				}
				log += ")";
			}
			
			return log;
		}

	}

	private static final Logger log = Logger.getLogger(SantaServlet.class);

	private static final int MAX_LOG_SIZE = 1000;

	private static SantaControlService	m_santaControlService;
	
	private static SantaSoundService	m_santaSoundService;
	
	private enum Activity {
		OPEN_MOUTH("/openMouth"),
		CLOSE_MOUTH("/closeMouth"),
		LEFT_TURN("/leftTurn"),
		RIGHT_TURN("/rightTurn"),
		STOP_TURN("/stopTurn"),
		START_DANCE("/startDance"),
		STOP_DANCE("/stopDance"),
		SPEAK("/speak"),
		SOUND("/sound"),
		LOG("/log");
		
		public final String commandPath;
		
		boolean matchesRequest(HttpServletRequest request) {
			return request.getPathInfo().equals(commandPath);
		}

		private Activity(final String commandPath) {
			this.commandPath = commandPath;
		}
	}
	
	private static LinkedList<LogEntry>	activityLog = new LinkedList<LogEntry>();
	
	@Override
	public synchronized void init(ServletConfig servletConfig) throws ServletException {
		super.init(servletConfig);
		
		if (m_santaControlService == null) {
			try {
				m_santaControlService = new SantaControlServiceImpl();
			} catch (Throwable t) {
				log.error("Unable to instantiate a SantaControlServiceImpl", t);
			}
		}
		
		if (m_santaSoundService == null) {
			m_santaSoundService = new SantaSoundServiceImpl();
		}
	}
	
	@Override
	public void service(ServletRequest servletRequest, ServletResponse response) throws IOException, ServletException {
		HttpServletRequest	request = (HttpServletRequest) servletRequest;
		HttpServletResponse	httpServletResponse = (HttpServletResponse) response;
		
		if (request.getMethod().equals("GET")) {
			if (Activity.OPEN_MOUTH.matchesRequest(request)) {
				logActivity(request, Activity.OPEN_MOUTH);
				m_santaControlService.openMouth();
			} else if (Activity.CLOSE_MOUTH.matchesRequest(request)) {
				logActivity(request, Activity.CLOSE_MOUTH);
				m_santaControlService.closeMouth();
			} else if (Activity.LEFT_TURN.matchesRequest(request)) {
				logActivity(request, Activity.LEFT_TURN);
				m_santaControlService.turnLeft();
			} else if (Activity.RIGHT_TURN.matchesRequest(request)) {
				logActivity(request, Activity.RIGHT_TURN);
				m_santaControlService.turnRight();
			} else if (Activity.STOP_TURN.matchesRequest(request)) {
				logActivity(request, Activity.STOP_TURN);
				m_santaControlService.turnStop();
			} else if (Activity.START_DANCE.matchesRequest(request)) {
				logActivity(request, Activity.START_DANCE);
				m_santaControlService.wiggleStart();
			} else if (Activity.STOP_DANCE.matchesRequest(request)) {
				logActivity(request, Activity.STOP_DANCE);
				m_santaControlService.wiggleStop();
			} else if (Activity.SPEAK.matchesRequest(request)) {
				boolean privit = true;	// By default it's private (not Tweeted)
				final String privateParam = request.getParameter("private");
				if (privateParam != null) {
					privit = Boolean.parseBoolean(privateParam);
				}
				final String voice = request.getParameter("voice");
				final String text = request.getParameter("text");
				
				if (!StringUtils.isEmpty(text)) {
					SantaVoice voiceFromTags = null;
					if (voice != null) {
						voiceFromTags = m_santaSoundService.getVoiceFromTags(voice.split(","));
					}
					logActivity(request, Activity.SPEAK, voiceFromTags==null?null:voiceFromTags.getId());
					m_santaSoundService.speak(text, privit, voiceFromTags==null?null:voiceFromTags.getId());
				} else {
					if (!StringUtils.isEmpty(voice)) {
						SantaVoice voiceFromTags = m_santaSoundService.getVoiceFromTags(voice);
						
						httpServletResponse.getWriter().print("Voice '" + voice + "' maps to: " + voiceFromTags.getName());
					} else {
						httpServletResponse.getWriter().println("Syntax: " + Activity.SPEAK.commandPath + "?text=<message>[&voice=<tags>][&private=true|false]");
						List<SantaVoice> voices = m_santaSoundService.getVoices();
						for (SantaVoice v : voices) {
							httpServletResponse.getWriter().println("  " + v.getId() + "=" + v.getName());
						}
					}
				}
			} else if (Activity.SOUND.matchesRequest(request)) {
				final String play = request.getParameter("play");
				if (!StringUtils.isEmpty(play)) {
					List<String> sounds = m_santaSoundService.getSounds(play);

					if (sounds != null && sounds.size() > 0) {
						m_santaSoundService.play(sounds.get(0));
						logActivity(request, Activity.SOUND, sounds.get(0));
					}
				} else {
					final String info = request.getParameter("info");
					if (!StringUtils.isEmpty(info)) {
						Map<String, String> soundTags = m_santaSoundService.getSoundTags(info);
						httpServletResponse.getWriter().print(new JSONObject(soundTags).toString());
					} else {
						httpServletResponse.getWriter().print("Syntax: " + Activity.SOUND.commandPath + "?[play|info]=<tags>");
					}
				}
			} else if (Activity.LOG.matchesRequest(request)) {
				response.setContentType("text/plain");
				synchronized(activityLog) {
					for (LogEntry entry : activityLog) {
						response.getWriter().println(entry);
					}
				}
			} else {
				log.info("Unsupported request path '" + request.getPathInfo() + "'");
				httpServletResponse.getWriter().print("Commands: [");
				for (Activity activity : Activity.values()) {
					if (Activity.values()[0] != activity) {
						httpServletResponse.getWriter().print("|");
					}
					httpServletResponse.getWriter().print(activity.commandPath);
				}
				httpServletResponse.getWriter().print("]");
			}
			httpServletResponse.setStatus(HttpServletResponse.SC_OK);
			return;
		}
		
		super.service(request, response);
	}

	private void logActivity(HttpServletRequest request, Activity activity, String... extraInfo) {
		LogEntry entry = new LogEntry(getRemoteAddress(request), activity, request.getQueryString(), extraInfo);
		
		synchronized(activityLog) {
			activityLog.add(0, entry);

			if (activityLog.size() > MAX_LOG_SIZE) {
				activityLog.removeLast();
			}
		}
	}

	private static String getRemoteAddress(HttpServletRequest request) {
		String addr = request.getHeader("X-Forwarded-For");
		if (!StringUtils.isEmpty(addr)) {
			addr = addr.split(",")[0];
		}
		
		if (StringUtils.isEmpty(addr)) {
			addr = request.getRemoteAddr();
		}
		
		if (log.isDebugEnabled()) {
			log.debug("Remote address " + addr);
		}
		
		return addr;
	}

	public void closeMouth() {
		log.debug("Close mouth");
		m_santaControlService.closeMouth();
	}

	public void faceForward() {
		m_santaControlService.faceForward();
	}

	public void faceLeft() {
		m_santaControlService.faceLeft();
	}

	public void faceRight() {
		m_santaControlService.faceRight();
	}

	public void openMouth() {
		log.debug("Open mouth");
		m_santaControlService.openMouth();
		log.debug("Mouth open");
	}

	public void turnLeft() {
		m_santaControlService.turnLeft();
	}

	public void turnRight() {
		m_santaControlService.turnRight();
	}

	public void turnStop() {
		m_santaControlService.turnStop();
	}

	public void wiggleStart() {
		m_santaControlService.wiggleStart();
	}

	public void wiggleStop() {
		m_santaControlService.wiggleStop();
	}

	public void speak(String text) {
		m_santaSoundService.speak(text);
	}
	
	public void speak(String text, boolean priveight, String voiceId) {
		m_santaSoundService.speak(text, priveight, voiceId);
	}
	
	public List<SantaVoice> getVoices() {
		ArrayList<SantaVoice> voices = new ArrayList<SantaVoice>();
		
		for (SantaVoice santaVoice : m_santaSoundService.getVoices()) {
			voices.add(new SantaVoice(santaVoice));
		}
		
		return voices;
	}
	
	public SantaVoice getVoiceFromTags(String... tags) {
		SantaVoice voiceFromTags = m_santaSoundService.getVoiceFromTags(tags);
		
		if (voiceFromTags != null) {
			return new SantaVoice(voiceFromTags);
		}
		
		return null;
	}
	
	public List<String> getSounds(String query) {
		return m_santaSoundService.getSounds(query);
	}
	
	public Map<String, String> getSoundTags(String query) {
		return m_santaSoundService.getSoundTags(query);
	}

	public void play(String soundBite) {
		m_santaSoundService.play(soundBite);
	}

	public void addSoundBite(byte[] audioData, String filename, String[] tags) throws SoundBiteException {
		m_santaSoundService.addSoundBite(audioData, filename, tags);
	}
}
